'use strict';

/* --------------------------------------------------------------
 slider_form.js 2016-12-12
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Controller Module For Slider Edit Form
 *
 * Handles the sliders details form operations.
 */
gx.controllers.module('slider_form', ['xhr', 'modal', 'loading_spinner'], function (data) {

    'use strict';

    // --------------------------------------------------------------------
    // VARIABLES
    // --------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Master data panel selector.
     *
     * @type {jQuery}
     */
    var $masterDataPanel = $this.find('.panel.master-data');

    /**
     * Slide panel containers (each language holds a container, that contains the slide panels).
     *
     * @type {jQuery}
     */
    var $tabContents = $this.find('.tab-pane');

    /**
     * Deleter select box.
     *
     * @type {jQuery}
     */
    var $imageDeleteSelect = $this.find('#delete_images');

    /**
     * Spinner Selector
     *
     * @type {jQuery}
     */
    var $spinner = null;

    /**
     * Do a refresh instead of redirecting?
     *
     * @type {Boolean}
     */
    var doRefresh = false;

    /**
     * Default Options
     *
     * @type {Object}
     */
    var defaults = {};

    /**
     * Final Options
     *
     * @type {Object}
     */
    var options = $.extend(true, {}, defaults, data);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // --------------------------------------------------------------------
    // FUNCTIONS
    // --------------------------------------------------------------------

    /**
     * Handles the form submit event.
     *
     * @param {jQuery.Event} event Triggered event.
     * @param {Object} data Event data.
     */
    function _onFormSubmit(event, data) {
        // Prevent the submit of the form.
        event.preventDefault();

        // Check refresh parameter.
        if (data && data.refresh) {
            doRefresh = true;
        }

        // Show loading spinner.
        $spinner = jse.libs.loading_spinner.show($this);

        // Upload files.
        _uploadFiles().then(_deleteFiles, _showFailMessage).then(_performSubmit, _showFailMessage).then(function () {
            return jse.libs.loading_spinner.hide($spinner);
        });
    }

    /**
     * Shows the submit error message modal.
     */
    function _showFailMessage() {
        // Message texts.
        var errorTitle = jse.core.lang.translate('FORM_SUBMIT_ERROR_MODAL_TITLE', 'sliders');
        var errorMessage = jse.core.lang.translate('FORM_SUBMIT_ERROR_MODAL_TEXT', 'sliders');

        // Show modal.
        jse.libs.modal.showMessage(errorTitle, errorMessage);

        // Hide loading spinner.
        if ($spinner) {
            jse.libs.loading_spinner.hide($spinner);
        }
    }

    /**
     * Performs the form submit AJAX request.
     */
    function _performSubmit() {
        // AJAX request options.
        var requestOptions = {
            url: $this.attr('action'),
            data: _getFormData()
        };

        jse.libs.xhr.post(requestOptions).done(function (response) {
            // URL to redirect to.
            var url = doRefresh ? 'admin.php?do=SlidersDetails&id=' + response.id : options.redirectUrl;

            // Prevent that the page unload prompt is displayed on save action.
            window.onbeforeunload = null;

            // Open overview page.
            window.open(url, '_self');
        }).fail(_showFailMessage);
    }

    /**
     * Returns the gathered form data.
     *
     * @return {Object} Form data.
     */
    function _getFormData() {
        // Form data object.
        var data = {
            id: $this.data('id')
        };

        // Extend form data object with all necessary properties.
        return $.extend(true, data, _getMasterData(), _getSlidesData());
    }

    /**
     * Returns the slider's master data.
     *
     * @return {Object} Slider master data.
     */
    function _getMasterData() {
        var name = $masterDataPanel.find('input[name="name"]').val();

        var speed = $masterDataPanel.find('input[name="speed"]').val();

        return { name: name, speed: speed };
    }

    /**
     * Returns the slides data by iterating over the tab content elements
     * which represent a container for each language.
     *
     * The returned object contains a property for each language.
     * The key is the language code and the value is an array containing
     * all slides information as collection.
     *
     * Example output:
     * {
     *   de: [{
     *     id: 1,
     *     thumbnail: 'My picture',
     *     ...
     *   }]
     * }
     *
     * @return {Object} Slides data.
     */
    function _getSlidesData() {
        // Slides data object.
        var slides = {};

        // Iterate over each slider panel container.
        $tabContents.each(function (index, element) {
            // Slide panel container element.
            var $slidePanelContainer = $(element);

            // Slide panel elements.
            var $slidePanels = $slidePanelContainer.find('.panel');

            // Get slide panel container language code.
            var languageCode = $slidePanelContainer.data('language');

            // Add property to slides data object,
            // which contains the language code as key and the slides data as value.
            slides[languageCode] = $.map($slidePanels, function (element) {
                return _getSlideData(element);
            });
        });

        return { slides: slides };
    }

    /**
     * Returns the data for a slide.
     *
     * @param {HTMLElement} slidePanel Slide panel element.
     *
     * @return {Object} Slide data.
     */
    function _getSlideData(slidePanel) {
        var $element = $(slidePanel);

        return {
            id: $element.data('id'),
            title: $element.find('input[name="title"]').val(),
            alt_text: $element.find('input[name="alt_text"]').val(),
            thumbnail: $element.find('select[name="thumbnail"]').val(),
            link: $element.find('input[name="link"]').val(),
            link_target: $element.find('select[name="link_target"]').val(),
            images: _getSlideImagesData(slidePanel)
        };
    }

    /**
     * Returns the slide images data.
     *
     * @param {HTMLElement} slidePanel Slide panel element.
     *
     * @return {Object} Slide images data.
     */
    function _getSlideImagesData(slidePanel) {
        // Image dropdown container elements (without thumbnail).
        var $imageDropdownContainers = $(slidePanel).find('.row.form-group').filter(function (index, element) {
            var $dropdown = $(element).find('.dropdown-input');

            return $dropdown.length && !$dropdown.is('[name="thumbnail"]');
        });

        // Iterate over each dropdown element and retrieve the data.
        return $.map($imageDropdownContainers, function (element) {
            // Dropdown container element.
            var $element = $(element);

            // Dropdown element.
            var $dropdown = $element.find('.dropdown-input');

            // Image data object.
            return {
                id: $element.data('id'),
                breakpoint: $dropdown.attr('name').replace('breakpoint_', ''),
                image: $dropdown.val(),
                areas: _getSlideImageAreaData(element)
            };
        });
    }

    /**
     * Returns the slide image area data.
     *
     * @param {HTMLElement} slideImageContainer Slide image configuration row element.
     *
     * @return {Object} Slide image area data.
     */
    function _getSlideImageAreaData(slideImageContainer) {
        // Slide image area data container list items.
        var $listItems = $(slideImageContainer).find('.image-map-data-list').children();

        // Iterate over each dropdown list item element and retrieve the data.
        return $.map($listItems, function (element) {
            // List item element.
            var $element = $(element);

            return {
                id: $element.data('id'),
                linkTitle: $element.data('linkTitle'),
                linkUrl: $element.data('linkUrl'),
                linkTarget: $element.data('linkTarget'),
                coordinates: $element.data('coordinates')
            };
        });
    }

    /**
     * Performs the images upload AJAX request.
     *
     * @return {jQuery.jqXHR} jQuery deferred object.
     */
    function _uploadFiles() {
        // Form data object.
        var formData = new FormData();

        // File inputs.
        var $fileInputs = $this.find(':file');

        // Append files to form data object.
        $fileInputs.each(function (index, element) {
            // File.
            var file = element.files[0];

            // Data entry key.
            var key = element.id + '[]';

            // Append file to form data object.
            if (file) {
                formData.append(key, file);
            }
        });

        // AJAX request options.
        var requestOptions = {
            url: options.imagesUploadUrl,
            data: formData,
            processData: false,
            contentType: false
        };

        // Perform AJAX request.
        return jse.libs.xhr.post(requestOptions);
    }

    /**
     * Performs the image deletion AJAX request.
     */
    function _deleteFiles() {
        // List of image file names.
        var fileNames = [];

        // List of thumbnail images.
        var thumbnailNames = [];

        // Fill the file names list.
        $imageDeleteSelect.children().each(function (index, element) {
            element.className === 'thumbnail' ? thumbnailNames.push(element.value) : fileNames.push(element.value);
        });

        // AJAX request options.
        var requestOptions = {
            url: options.imagesDeleteUrl,
            data: { file_names: fileNames, thumbnail_names: thumbnailNames }
        };

        return jse.libs.xhr.post(requestOptions);
    }

    // --------------------------------------------------------------------
    // INITIALIZATION
    // --------------------------------------------------------------------

    module.init = function (done) {
        // Listen to form submit event.
        $this.on('submit', _onFormSubmit);

        // Finish initialization.
        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
